#!/usr/bin/env python3
"""
ניתוח יומי של ביצועי הבוט
מציג סטטיסטיקות מהלוג של היום
"""

import json
import os
from datetime import datetime
from pathlib import Path


def analyze_today():
    """מנתח את הלוג של היום"""
    
    # מציאת קובץ הלוג של היום
    logs_dir = Path("/home/tradingbot/logs")
    today = datetime.now().strftime("%Y-%m-%d")
    log_file = logs_dir / f"bot_log_{today}.json"
    
    if not log_file.exists():
        print(f"❌ לא נמצא לוג להיום: {log_file}")
        return
    
    print(f"\n{'='*80}")
    print(f"📊 Daily Analysis - {today}")
    print(f"{'='*80}\n")
    
    # טעינת הלוג
    with open(log_file, 'r', encoding='utf8') as f:
        try:
            cycles = json.load(f)
        except json.JSONDecodeError:
            print("❌ שגיאה בקריאת הלוג")
            return
    
    print(f"📈 Total Cycles: {len(cycles):,}")
    
    # ניתוח טריידים
    trades = []
    position_open = None
    
    for cycle in cycles:
        decision = cycle.get('decision', {})
        exec_info = decision.get('exec_info', {})
        exec_side = exec_info.get('exec_side', '')
        
        if exec_side == 'OPEN_LONG':
            position_open = {
                'entry_price': exec_info.get('exec_price'),
                'timestamp': cycle.get('timestamp_readable', '')
            }
        
        elif exec_side == 'CLOSE_LONG' and position_open:
            pnl = exec_info.get('pnl_pct_exec', 0)
            hold_time = exec_info.get('hold_time_sec', 0)
            
            trades.append({
                'pnl': pnl,
                'hold_time': hold_time,
                'reason': decision.get('reason', ''),
                'entry': position_open['timestamp'],
                'exit': cycle.get('timestamp_readable', '')
            })
            position_open = None
    
    if not trades:
        print("\n⚠️  אין טריידים שהושלמו היום\n")
        if position_open:
            print("📍 יש פוזיציה פתוחה כרגע")
        return
    
    # חישוב סטטיסטיקות
    total_trades = len(trades)
    pnls = [t['pnl'] for t in trades]
    wins = [p for p in pnls if p > 0]
    losses = [p for p in pnls if p <= 0]
    
    total_pnl = sum(pnls)
    win_rate = (len(wins) / total_trades * 100) if total_trades > 0 else 0
    avg_pnl = sum(pnls) / total_trades if total_trades > 0 else 0
    
    best_trade = max(pnls)
    worst_trade = min(pnls)
    
    avg_hold = sum(t['hold_time'] for t in trades) / total_trades if total_trades > 0 else 0
    
    # הצגת תוצאות
    print(f"\n{'─'*80}")
    print(f"💼 TRADING SUMMARY")
    print(f"{'─'*80}")
    print(f"Total Trades:     {total_trades}")
    print(f"✅ Wins:          {len(wins)}")
    print(f"❌ Losses:        {len(losses)}")
    print(f"🎯 Win Rate:      {win_rate:.1f}%")
    print(f"{'─'*80}")
    print(f"💰 Total PnL:     {total_pnl:+.2f}%")
    print(f"📊 Average PnL:   {avg_pnl:+.3f}%")
    print(f"🏆 Best Trade:    {best_trade:+.2f}%")
    print(f"💔 Worst Trade:   {worst_trade:+.2f}%")
    print(f"⏱️  Avg Hold Time: {avg_hold/60:.1f} min")
    print(f"{'─'*80}\n")
    
    # פירוט טריידים
    if total_trades <= 10:
        print("📋 All Trades:")
        print(f"{'─'*80}")
        for i, trade in enumerate(trades, 1):
            emoji = "✅" if trade['pnl'] > 0 else "❌"
            print(f"{i}. {emoji} {trade['pnl']:+.2f}% | {trade['hold_time']/60:.1f}min | {trade['reason']}")
            print(f"   Entry: {trade['entry']} → Exit: {trade['exit']}")
        print(f"{'─'*80}\n")
    
    # סיבות לסגירה
    print("📊 Exit Reasons:")
    reasons = {}
    for t in trades:
        reason = t['reason']
        reasons[reason] = reasons.get(reason, 0) + 1
    
    for reason, count in sorted(reasons.items(), key=lambda x: x[1], reverse=True):
        print(f"   {reason}: {count} ({count/total_trades*100:.1f}%)")
    
    print(f"\n{'='*80}\n")


def main():
    try:
        analyze_today()
    except Exception as e:
        print(f"\n❌ Error: {e}\n")


if __name__ == "__main__":
    main()
