import numpy as np
from indicators import calc_trend_score, calc_mf_stats


class TradingEngine:
    def __init__(self):
        pass


    # =====================================================================
    #  MAIN CYCLE
    # =====================================================================
    def process_cycle(self, state, decision):

        has_position = state.get("has_position", False)
        pnl_pct = decision.get("pnl_pct", 0)

        buy_ratio = state.get("buy_ratio", 0)
        ratio_5m = state.get("ratio_5m", 0)
        ratio_15m = state.get("ratio_15m", 0)
        cvd = state.get("cvd", 0)
        whale = state.get("whale", 0)
        depth_ratio = state.get("depth_ratio", 0)
        smooth = state.get("smooth", 0)
        price_change_5m = state.get("price_change_pct_5m", 0)

        f_mempool, f_fee, f_block = state.get("onchain", [0,0,0])

        trend_score = calc_trend_score(
            smooth,
            buy_ratio,
            cvd,
            whale,
            f_mempool,
            f_fee,
            f_block
        )

        # =====================================================================
        #  SELL LOGIC (simple & safe)
        # =====================================================================
        if has_position:

            if pnl_pct >= 0.01:
                return {
                    "action": "SELL",
                    "reason": "TP_hit",
                    "pnl_pct": pnl_pct,
                    "trend_score": trend_score
                }

            if pnl_pct <= -0.01:
                return {
                    "action": "SELL",
                    "reason": "SL_hit",
                    "pnl_pct": pnl_pct,
                    "trend_score": trend_score
                }

            return {
                "action": "WAIT",
                "reason": "holding",
                "pnl_pct": pnl_pct,
                "trend_score": trend_score
            }

        # =====================================================================
        #  BUY LOGIC v2 (From real trade analysis)
        # =====================================================================

        # 1. Depth filter (bad zone)
        if 1 <= depth_ratio <= 5:
            return {"action": "WAIT", "reason": "depth_bad"}

        # 2. Smooth too high
        if smooth > 0.55:
            return {"action": "WAIT", "reason": "smooth_high"}

        # 3. Momentum low
        if price_change_5m < 0.05:
            return {"action": "WAIT", "reason": "momentum_low"}

        # 4. Strong indicators combined
        strong_buy = (
            buy_ratio > 0.76 and
            ratio_5m > 0.76 and
            ratio_15m > 0.76 and
            cvd > 2.0 and
            whale > 0.8
        )

        # 5. Trend score integration
        trend_needed = (
            trend_score > 65 or
            (trend_score > 55 and buy_ratio > 0.80)
        )

        if strong_buy and trend_needed:
            return {
                "action": "BUY",
                "reason": "v2_strong_entry",
                "trend_score": trend_score,
                "buy_ratio": buy_ratio,
                "cvd": cvd,
                "whale": whale,
                "depth_ratio": depth_ratio
            }

        return {"action": "WAIT", "reason": "no_conditions"}



# =====================================================================
#  detect_signal FOR run.py compatibility
# =====================================================================
def detect_signal(state):
    engine = TradingEngine()
    decision = {}   # run.py does not provide decision
    return engine.process_cycle(state, decision)
