import time
import json
import math
import asyncio
import aiohttp
import requests
import numpy as np
from collections import deque

from engine import detect_signal
from indicators import calc_mf_stats
from logger import append_log
from mempool import fetch_mempool, normalize_mempool


# ================================================================
# GLOBAL STATE
# ================================================================

PRICE_WINDOW = deque(maxlen=120)   # ~2 דקות אם לולאה כל שנייה

HAS_POSITION = False
BUY_PRICE = None
ENTRY_TS = None  # זמן פתיחת הפוזיציה

# ====== מעקב טריידים ======
TOTAL_TRADES = 0
WINS = 0
LOSSES = 0
TOTAL_PNL = 0.0
BEST_TRADE = 0.0
WORST_TRADE = 0.0
# ============================


# ================================================================
# HELPERS
# ================================================================

def get_aggtrades(symbol: str = "BTCUSDT", limit: int = 500):
    """
    מביא aggTrades מבינאנס (SPOT או FUTURES – תלוי ב-URL שתבחר).
    כרגע: SPOT.
    """
    url = f"https://api.binance.com/api/v3/aggTrades?symbol={symbol}&limit={limit}"
    try:
        r = requests.get(url, timeout=2)
        r.raise_for_status()
        data = r.json()
    except Exception as e:
        print(f"Error fetching aggTrades: {e}")
        return []

    trades = []
    for t in data:
        # מבנה מקורי: {"p": "...", "q": "...", "m": bool}
        price = float(t["p"])
        qty = float(t["q"])
        is_buyer_maker = bool(t["m"])  # m=True => SELL-side aggressive, אז isBuyerMaker=True
        trades.append({
            "p": price,
            "q": qty,
            "isBuyerMaker": is_buyer_maker,
        })
    return trades


def get_orderbook(symbol: str = "BTCUSDT", limit: int = 50):
    """
    ספר פקודות: עומק + best-level ratio.
    """
    url = f"https://api.binance.com/api/v3/depth?symbol={symbol}&limit={limit}"
    try:
        r = requests.get(url, timeout=2)
        r.raise_for_status()
        ob = r.json()
    except Exception as e:
        print(f"Error fetching orderbook: {e}")
        return 1.0, 1.0  # fallback

    try:
        bids = sum(float(x[1]) for x in ob["bids"])
        asks = sum(float(x[1]) for x in ob["asks"])
        best_bid = float(ob["bids"][0][0])
        best_ask = float(ob["asks"][0][0])

        depth_ratio = bids / asks if asks > 0 else 1.0
        best_ratio = best_bid / best_ask if best_ask > 0 else 1.0
    except Exception as e:
        print(f"Error parsing orderbook: {e}")
        depth_ratio = 1.0
        best_ratio = 1.0

    return depth_ratio, best_ratio


def compute_smooth(prices: list[float]) -> float:
    """
    החלקה / שיפוע נורמלי על המחירים — רק לאינדיקציה (לא חובה מדויק).
    """
    if len(prices) < 5:
        return 0.0

    x = np.arange(len(prices))
    y = np.array(prices, dtype=float)

    try:
        # פולינום מדרגה 1 – שיפוע
        m, b = np.polyfit(x, y, 1)
        # ננרמל טיפה ע"י טנגנס-היפרבולי
        smooth = math.tanh(m / (y.mean() * 0.0001 + 1e-8))
    except Exception:
        smooth = 0.0

    return float(smooth)


def print_trade_summary():
    """הדפסת סיכום טריידים"""
    global TOTAL_TRADES, WINS, LOSSES, TOTAL_PNL, BEST_TRADE, WORST_TRADE
    
    win_rate = (WINS / TOTAL_TRADES * 100) if TOTAL_TRADES > 0 else 0
    avg_pnl = (TOTAL_PNL / TOTAL_TRADES) if TOTAL_TRADES > 0 else 0
    
    print("\n" + "="*80)
    print("📊 TRADING SUMMARY")
    print("="*80)
    print(f"Total Trades: {TOTAL_TRADES}")
    print(f"✅ Wins: {WINS} | ❌ Losses: {LOSSES}")
    print(f"🎯 Win Rate: {win_rate:.1f}%")
    print(f"💰 Total PnL: {TOTAL_PNL:+.2f}%")
    print(f"📈 Average PnL: {avg_pnl:+.3f}%")
    print(f"🏆 Best Trade: {BEST_TRADE:+.2f}%")
    print(f"💔 Worst Trade: {WORST_TRADE:+.2f}%")
    print("="*80 + "\n")


# ================================================================
# ONE CYCLE
# ================================================================

async def run_cycle(session, symbol: str = "BTCUSDT"):
    global HAS_POSITION, BUY_PRICE, ENTRY_TS
    global TOTAL_TRADES, WINS, LOSSES, TOTAL_PNL, BEST_TRADE, WORST_TRADE

    t0 = time.time()

    # --------- AGGTRADES ----------
    trades = get_aggtrades(symbol=symbol)
    if not trades:
        print("No trades")
        return

    price = trades[-1]["p"]
    PRICE_WINDOW.append(price)
    smooth = compute_smooth(list(PRICE_WINDOW))

    # --------- ORDERBOOK ----------
    depth_ratio, best_level_ratio = get_orderbook(symbol=symbol)

    # --------- MF / FLOW ----------
    mf = calc_mf_stats(trades)
    buy_ratio = mf.get("buy_ratio", 0.5)
    ratio_1m = mf.get("ratio_1m", buy_ratio)
    ratio_5m = mf.get("ratio_5m", buy_ratio)
    ratio_15m = mf.get("ratio_15m", ratio_5m)
    cvd = mf.get("cvd", 0.0)
    whale = mf.get("whale", 0.0)
    price_change_5m = mf.get("price_change_pct_5m", 0.0)
    vol_flag = mf.get("vol_flag", "NORMAL")
    spoof = mf.get("spoof", {"bid_spoof": False, "ask_spoof": False})
    iceberg = mf.get("iceberg", {"iceberg_buy": False, "iceberg_sell": False})
    mf_stats = mf.get("mf_stats", {})

    # --- MEMPOOL / ONCHAIN ---
    try:
        mem = await fetch_mempool(session)
        onchain = normalize_mempool(mem)
    except Exception as e:
        print(f"Error fetching mempool: {e}")
        onchain = [0.0, 0.0, 0.0]

    # --- POSITION AGE (שימושי ללוג בלבד) ---
    position_age = 0.0
    if HAS_POSITION and ENTRY_TS is not None:
        position_age = time.time() - ENTRY_TS

    # --------- STATE לבוט / לוג ----------
    bot_state = {
        "price": price,
        "smooth": smooth,
        "buy_ratio": buy_ratio,
        "cvd": cvd,
        "whale": whale,
        "depth_ratio": depth_ratio,
        "best_level_ratio": best_level_ratio,
        "ratio_1m": ratio_1m,
        "ratio_5m": ratio_5m,
        "ratio_15m": ratio_15m,
        "price_change_pct_5m": price_change_5m,
        "mf_stats": mf_stats,
        "onchain": onchain,
        "vol_price": vol_flag,
        "spoof": spoof,
        "iceberg": iceberg,
        "has_position": HAS_POSITION,
        "buy_price": BUY_PRICE,
        "position_age": position_age,
    }

    # --------- ENGINE DECISION ----------
    decision = detect_signal(bot_state)

    # --------- EXECUTION LOGIC (פוזיציה יחידה) ----------
    executed = False
    exec_info = {}

    action = decision.get("action")

    if action == "BUY":
        if not HAS_POSITION:
            HAS_POSITION = True
            BUY_PRICE = price
            ENTRY_TS = time.time()
            executed = True
            exec_info = {
                "exec_side": "OPEN_LONG",
                "exec_price": price,
            }
            print(f"\n🚀 OPENED POSITION at ${price:,.2f}")
        else:
            exec_info = {
                "exec_side": "SKIP_BUY_ALREADY_IN_POSITION",
                "exec_price": price,
            }

    elif action == "SELL":
        if HAS_POSITION and BUY_PRICE is not None:
            pnl_pct_exec = (price - BUY_PRICE) / BUY_PRICE * 100.0
            executed = True
            exec_info = {
                "exec_side": "CLOSE_LONG",
                "exec_price": price,
                "pnl_pct_exec": pnl_pct_exec,
                "hold_time_sec": (time.time() - ENTRY_TS) if ENTRY_TS else None,
            }
            
            # ====== עדכון סטטיסטיקות ======
            TOTAL_TRADES += 1
            TOTAL_PNL += pnl_pct_exec
            
            if pnl_pct_exec > 0:
                WINS += 1
                emoji = "✅"
            else:
                LOSSES += 1
                emoji = "❌"
            
            if pnl_pct_exec > BEST_TRADE:
                BEST_TRADE = pnl_pct_exec
            
            if pnl_pct_exec < WORST_TRADE:
                WORST_TRADE = pnl_pct_exec
            
            # הדפסת הטריד שנסגר
            print(f"\n{emoji} CLOSED POSITION at ${price:,.2f}")
            print(f"   Entry: ${BUY_PRICE:,.2f} | Exit: ${price:,.2f}")
            print(f"   PnL: {pnl_pct_exec:+.2f}%")
            print(f"   Hold Time: {exec_info['hold_time_sec']:.0f}s ({exec_info['hold_time_sec']/60:.1f}min)")
            print(f"   Reason: {decision.get('reason')}")
            
            # הדפסת סיכום כללי
            print_trade_summary()
            # ===============================
            
            HAS_POSITION = False
            BUY_PRICE = None
            ENTRY_TS = None
        else:
            exec_info = {
                "exec_side": "SKIP_SELL_NO_POSITION",
                "exec_price": price,
            }

    # עדכון decision עם מידע ביצוע
    decision["executed"] = executed
    decision["exec_info"] = exec_info

    # --------- LOG לקובץ ----------
    append_log({
        "state": bot_state,
        "decision": decision,
        "ts": time.time(),
    })

    # --------- PRINT לקונסול ----------
    summary = {
        "price": round(price, 2),
        "action": action,
        "reason": decision.get("reason"),
        "position_age": int(position_age),
        "has_position": HAS_POSITION,
        "buy_price": round(BUY_PRICE, 2) if BUY_PRICE else None,
        "executed": executed,
    }
    print(">>>", summary)
    print(f"cycle time: {time.time() - t0:.4f}s")


# ================================================================
# MAIN LOOP
# ================================================================

async def main():
    print("\n" + "🤖"*40)
    print("BOT STARTED - LIVE TRADING WITH STATISTICS")
    print("🤖"*40 + "\n")
    
    async with aiohttp.ClientSession() as session:
        while True:
            await run_cycle(session, symbol="BTCUSDT")
            await asyncio.sleep(1)


if __name__ == "__main__":
    asyncio.run(main())